<?php
/**
 * Plugin Name: Dima Ajax Cart Pro
 * Description: WooCommerce cart sidebar (Tailwind-friendly) with a simple widget and shortcode.
 * Version: 1.0.0
 * Author: Dima
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

define( 'DACP_PATH', plugin_dir_path( __FILE__ ) );
define( 'DACP_URL', plugin_dir_url( __FILE__ ) );

// استایل badge سبد + انیمیشن پالس + مودال
add_action( 'wp_head', function () { ?>
    <style>
        /* خود دایره‌ی تعداد سبد */
        .cart-counter-badge{
            background: #2563eb !important;  /* آبی خوش‌دست */
            color: #fff !important;
            min-width: 22px;
            min-height: 22px;
            font-size: 11px;
            line-height: 1;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
        }

        /* وقتی کلاس پالس اضافه می‌کنیم */
        .cart-counter-badge.dacp-badge-pulse{
            animation: dacpBadgePulse .5s ease-out;
        }

        @keyframes dacpBadgePulse{
            0%   { transform: scale(1); }
            50%  { transform: scale(1.35); }
            100% { transform: scale(1); }
        }

        /* انیمیشن دکمه افزودن به سبد */
        .add_to_cart_button.adding,
        .single_add_to_cart_button.adding {
            position: relative;
            pointer-events: none;
            opacity: 0.7;
        }

        .add_to_cart_button.adding::after,
        .single_add_to_cart_button.adding::after {
            content: '';
            position: absolute;
            width: 20px;
            height: 20px;
            top: 50%;
            left: 50%;
            margin-left: -10px;
            margin-top: -10px;
            border: 2px solid #ffffff;
            border-top-color: transparent;
            border-radius: 50%;
            animation: dacpSpin 0.6s linear infinite;
        }

        @keyframes dacpSpin {
            to { transform: rotate(360deg); }
        }

        /* مودال موفقیت */
        #dacp-success-modal {
            position: fixed;
            inset: 0;
            z-index: 10002;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            visibility: hidden;
            transition: opacity 0.3s ease, visibility 0.3s ease;
        }

        #dacp-success-modal.show {
            opacity: 1;
            visibility: visible;
        }

        #dacp-success-modal-overlay {
            position: absolute;
            inset: 0;
            background: rgba(0, 0, 0, 0.5);
            backdrop-filter: blur(4px);
        }

        #dacp-success-modal-content {
            position: relative;
            background: #fff;
            border-radius: 1rem;
            padding: 2rem;
            max-width: 400px;
            width: 90%;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
            transform: scale(0.9) translateY(20px);
            transition: transform 0.3s ease;
            text-align: center;
        }

        #dacp-success-modal.show #dacp-success-modal-content {
            transform: scale(1) translateY(0);
        }

        .dacp-success-icon {
            width: 64px;
            height: 64px;
            margin: 0 auto 1rem;
            background: #10b981;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            animation: dacpSuccessBounce 0.6s ease;
        }

        .dacp-success-icon svg {
            width: 32px;
            height: 32px;
            color: #fff;
        }

        @keyframes dacpSuccessBounce {
            0% { transform: scale(0); }
            50% { transform: scale(1.1); }
            100% { transform: scale(1); }
        }

        .dacp-success-title {
            font-size: 1.25rem;
            font-weight: 700;
            color: #111827;
            margin-bottom: 0.5rem;
        }

        .dacp-success-message {
            color: #6b7280;
            font-size: 0.875rem;
            margin-bottom: 1.5rem;
        }

        .dacp-success-buttons {
            display: flex;
            gap: 0.75rem;
            flex-direction: column;
        }

        .dacp-success-btn {
            padding: 0.75rem 1.5rem;
            border-radius: 0.5rem;
            font-weight: 600;
            font-size: 0.875rem;
            transition: all 0.2s ease;
            cursor: pointer;
            border: none;
            display: block;
            width: 100%;
        }

        .dacp-success-btn-continue {
            background: #f3f4f6;
            color: #374151;
        }

        .dacp-success-btn-continue:hover {
            background: #e5e7eb;
        }

        .dacp-success-btn-cart {
            background: #2563eb;
            color: #fff;
        }

        .dacp-success-btn-cart:hover {
            background: #1d4ed8;
        }

        /* انیمیشن برای محصول اضافه شده */
        @keyframes dacpProductAdded {
            0% {
                transform: scale(0.8);
                opacity: 0;
            }
            50% {
                transform: scale(1.05);
            }
            100% {
                transform: scale(1);
                opacity: 1;
            }
        }

        .dacp-product-added {
            animation: dacpProductAdded 0.5s ease;
        }
    </style>
<?php } );

// اضافه کردن HTML مودال موفقیت به footer
add_action('wp_footer', function() {
    $cart_url = function_exists('wc_get_cart_url') ? wc_get_cart_url() : '/cart/';
    ?>
    <div id="dacp-success-modal">
        <div id="dacp-success-modal-overlay"></div>
        <div id="dacp-success-modal-content">
            <div class="dacp-success-icon">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                </svg>
            </div>
            <h3 class="dacp-success-title">محصول با موفقیت اضافه شد!</h3>
            <p class="dacp-success-message">محصول به سبد خرید شما اضافه شد</p>
            <div class="dacp-success-buttons">
                <button type="button" class="dacp-success-btn dacp-success-btn-cart" data-cart-url="<?php echo esc_url($cart_url); ?>">مشاهده سبد خرید</button>
                <button type="button" class="dacp-success-btn dacp-success-btn-continue dacp-success-close">ادامه خرید</button>
            </div>
        </div>
    </div>
    <?php
});

// Bootstrap safely after WooCommerce is loaded
add_action('plugins_loaded', function(){
    if ( ! class_exists( 'WooCommerce' ) ) {
        add_action('admin_notices', function(){
            echo '<div class="notice notice-error"><p>Dima Ajax Cart Pro: WooCommerce باید فعال باشد.</p></div>';
        });
        return;
    }

    // Enqueue assets
    add_action( 'wp_enqueue_scripts', function() {
        // Enqueue CSS
        //wp_enqueue_style( 'dacp-cart', DACP_URL . 'assets/css/cart.css', [], '1.0.0' );
        
        // Enqueue JS با priority بالا برای اجرای زودهنگام
        wp_enqueue_script( 'dacp-cart', DACP_URL . 'assets/js/cart.js', ['jquery'], '1.0.1', true );
        
        // Localize script with AJAX URL
        wp_localize_script( 'dacp-cart', 'dacpAjax', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('dacp_nonce'),
            'wc_ajax_url' => WC_AJAX::get_endpoint('%%endpoint%%'),
            'is_single_product' => is_product() // بررسی اینکه آیا در صفحه تک محصول هستیم
        ]);
        
        // اضافه کردن wc_add_to_cart_params اگر وجود ندارد
        if ( ! wp_script_is('wc-add-to-cart', 'enqueued') ) {
            wp_localize_script( 'dacp-cart', 'wc_add_to_cart_params', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'wc_ajax_url' => WC_AJAX::get_endpoint('%%endpoint%%'),
                'i18n_view_cart' => esc_attr__('View cart', 'woocommerce'),
                'cart_url' => wc_get_cart_url(),
                'is_cart' => is_cart(),
                'cart_redirect_after_add' => get_option('woocommerce_cart_redirect_after_add'),
                'security' => wp_create_nonce('woocommerce-cart')
            ]);
        }
    });

    // AJAX: Remove item from cart
    add_action('wp_ajax_dacp_remove_item', 'dacp_remove_cart_item');
    add_action('wp_ajax_nopriv_dacp_remove_item', 'dacp_remove_cart_item');
    
    // AJAX: Update item quantity
    add_action('wp_ajax_dacp_update_qty', 'dacp_update_cart_qty');
    add_action('wp_ajax_nopriv_dacp_update_qty', 'dacp_update_cart_qty');
    
    // استفاده از handler استاندارد WooCommerce برای add_to_cart
    // handler سفارشی را غیرفعال می‌کنیم تا از handler استاندارد WooCommerce استفاده شود
    // handler استاندارد WooCommerce خودش variation_id و variation را مدیریت می‌کند
    // فقط برای موارد خاص که نیاز به handler سفارشی داریم، از این استفاده می‌کنیم
    // add_action('wc_ajax_add_to_cart', 'dacp_ajax_add_to_cart');
    // add_action('wp_ajax_woocommerce_ajax_add_to_cart', 'dacp_ajax_add_to_cart');
    // add_action('wp_ajax_nopriv_woocommerce_ajax_add_to_cart', 'dacp_ajax_add_to_cart');

    // Shortcode
    add_shortcode('dima_cart_pro', function(){
        if ( function_exists('is_cart') && (is_cart() || is_checkout()) ) return '';
        return '<div class="dacp-wrapper">' . dacp_render_button() . '</div>' . dacp_render_drawer();
    });

    // Widget
    add_action('widgets_init', function(){ register_widget('Dima_Ajax_Cart_Pro_Widget'); });
    
    // اضافه کردن fragment برای mini cart
    add_filter('woocommerce_add_to_cart_fragments', 'dacp_cart_fragments');
    
    // تبدیل لینک‌های افزودن به سبد خرید به form برای Ajax
    add_filter('woocommerce_loop_add_to_cart_link', 'dacp_convert_add_to_cart_link_to_form', 10, 3);
});

// تبدیل لینک افزودن به سبد خرید به form
function dacp_convert_add_to_cart_link_to_form( $link, $product, $args ) {
    // فقط برای محصولات ساده
    if ( ! $product->is_type('simple') || ! $product->is_purchasable() || ! $product->is_in_stock() ) {
        return $link;
    }
    
    $product_id = $product->get_id();
    $button_class = isset($args['class']) ? $args['class'] : 'button';
    
    // ساخت form به جای لینک
    $form = sprintf(
        '<form class="cart" action="%s" method="post" enctype="multipart/form-data">
            <input type="hidden" name="add-to-cart" value="%d" />
            <input type="hidden" name="quantity" value="1" />
            <button type="submit" name="add-to-cart" value="%d" class="%s add_to_cart_button product_type_%s" data-product_id="%d">
                %s
            </button>
        </form>',
        esc_url( apply_filters( 'woocommerce_add_to_cart_form_action', $product->get_permalink() ) ),
        esc_attr( $product_id ),
        esc_attr( $product_id ),
        esc_attr( $button_class ),
        esc_attr( $product->get_type() ),
        esc_attr( $product_id ),
        esc_html( $product->add_to_cart_text() )
    );
    
    return $form;
}

// AJAX Handler: Add to cart (مطابق با متد استاندارد WooCommerce)
function dacp_ajax_add_to_cart() {
    ob_start();
    
    try {
        // بررسی product_id
        if ( ! isset( $_POST['product_id'] ) ) {
            wp_send_json( array(
                'error' => true,
                'message' => __( 'Product ID is required.', 'woocommerce' ),
            ) );
            return;
        }
        
        $product_id        = apply_filters( 'woocommerce_add_to_cart_product_id', absint( $_POST['product_id'] ) );
        $product           = wc_get_product( $product_id );
        $quantity          = empty( $_POST['quantity'] ) ? 1 : wc_stock_amount( wp_unslash( $_POST['quantity'] ) );
        $variation_id      = isset( $_POST['variation_id'] ) ? absint( $_POST['variation_id'] ) : 0;
        $variation         = isset( $_POST['variation'] ) && is_array( $_POST['variation'] ) ? $_POST['variation'] : array();
        
        // لاگ‌گذاری برای دیباگ
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DACP Add to Cart - Product ID: ' . $product_id );
            error_log( 'DACP Add to Cart - Variation ID: ' . $variation_id );
            error_log( 'DACP Add to Cart - Variation: ' . print_r( $variation, true ) );
            error_log( 'DACP Add to Cart - POST Data: ' . print_r( $_POST, true ) );
        }
        
        // اگر variation_id از POST ارسال شده، از آن استفاده کن (برای محصولات متغیر)
        if ( $variation_id > 0 ) {
            $variation_product = wc_get_product( $variation_id );
            if ( $variation_product && $variation_product->is_type( 'variation' ) ) {
                // product_id باید parent product باشد
                $product_id = $variation_product->get_parent_id();
                $product    = wc_get_product( $product_id );
                
                // اگر variation attributes ارسال نشده، از variation product بگیر
                if ( empty( $variation ) ) {
                    $variation = $variation_product->get_variation_attributes();
                } else {
                    // اگر variation attributes ارسال شده، آنها را با variation attributes از variation product merge کن
                    $variation_from_product = $variation_product->get_variation_attributes();
                    $variation = array_merge( $variation_from_product, $variation );
                }
            }
        }
        
        // بررسی محصولات متغیر (اگر product_id یک variation باشد)
        if ( $product && $product->is_type( 'variation' ) ) {
            $variation_id = $product_id;
            $product_id   = $product->get_parent_id();
            $variation    = $product->get_variation_attributes();
            $product      = wc_get_product( $product_id );
        }
        
        $product_status = get_post_status( $variation_id > 0 ? $variation_id : $product_id );
        
        // بررسی اینکه آیا محصول وجود دارد
        if ( ! $product ) {
            wp_send_json( array(
                'error'       => true,
                'product_url' => apply_filters( 'woocommerce_cart_redirect_after_error', get_permalink( $product_id ), $product_id ),
                'message'     => __( 'Product not found.', 'woocommerce' ),
            ) );
            return;
        }
        
        // بررسی status محصول
        if ( 'publish' !== $product_status ) {
            wp_send_json( array(
                'error'       => true,
                'product_url' => apply_filters( 'woocommerce_cart_redirect_after_error', get_permalink( $product_id ), $product_id ),
                'message'     => __( 'Product is not available.', 'woocommerce' ),
            ) );
            return;
        }
        
        // برای محصولات متغیر، بررسی اینکه variation انتخاب شده است
        if ( $product->is_type( 'variable' ) ) {
            if ( ! $variation_id || $variation_id <= 0 ) {
                wp_send_json( array(
                    'error'       => true,
                    'product_url' => apply_filters( 'woocommerce_cart_redirect_after_error', get_permalink( $product_id ), $product_id ),
                    'message'     => __( 'Please choose product options.', 'woocommerce' ),
                ) );
                return;
            }
            
            // بررسی اینکه variation وجود دارد
            $variation_product = wc_get_product( $variation_id );
            if ( ! $variation_product ) {
                wp_send_json( array(
                    'error'       => true,
                    'product_url' => apply_filters( 'woocommerce_cart_redirect_after_error', get_permalink( $product_id ), $product_id ),
                    'message'     => __( 'Variation not found.', 'woocommerce' ),
                ) );
                return;
            }
            
            // بررسی موجودی variation
            if ( ! $variation_product->is_in_stock() ) {
                wp_send_json( array(
                    'error'       => true,
                    'product_url' => apply_filters( 'woocommerce_cart_redirect_after_error', get_permalink( $product_id ), $product_id ),
                    'message'     => __( 'This variation is out of stock.', 'woocommerce' ),
                ) );
                return;
            }
            
            // برای محصولات متغیر، باید از variation product استفاده کنیم
            $product_to_add = $variation_product;
        } else {
            $product_to_add = $product;
        }
        
        // استفاده از validation استاندارد WooCommerce
        $passed_validation = apply_filters( 'woocommerce_add_to_cart_validation', true, $product_id, $quantity, $variation_id, $variation );
        
        if ( ! $passed_validation ) {
            // بررسی notices برای پیام خطا
            $notices = wc_get_notices( 'error' );
            $error_message = ! empty( $notices ) ? wp_strip_all_tags( $notices[0]['notice'] ) : __( 'Validation failed.', 'woocommerce' );
            wc_clear_notices();
            
            wp_send_json( array(
                'error'       => true,
                'product_url' => apply_filters( 'woocommerce_cart_redirect_after_error', get_permalink( $product_id ), $product_id ),
                'message'     => $error_message,
            ) );
            return;
        }
        
        // افزودن به سبد خرید با استفاده از try-catch برای catch کردن exception ها
        // WooCommerce خودش validation می‌کند و اگر مشکل داشته باشد، exception می‌اندازد
        try {
            // لاگ‌گذاری قبل از افزودن به سبد خرید
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( 'DACP Add to Cart - Before add_to_cart: product_id=' . $product_id . ', variation_id=' . $variation_id . ', quantity=' . $quantity );
                error_log( 'DACP Add to Cart - Variation attributes: ' . print_r( $variation, true ) );
            }
            
            $cart_item_key = WC()->cart->add_to_cart( $product_id, $quantity, $variation_id, $variation );
            
            if ( false !== $cart_item_key && ! is_wp_error( $cart_item_key ) ) {
                
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    error_log( 'DACP Add to Cart - Success: cart_item_key=' . $cart_item_key );
                }
                
                do_action( 'woocommerce_ajax_added_to_cart', $product_id );
                
                if ( 'yes' === get_option( 'woocommerce_cart_redirect_after_add' ) ) {
                    wc_add_to_cart_message( array( $product_id => $quantity ), true );
                }
                
                WC_AJAX::get_refreshed_fragments();
                
            } else {
                
                // بررسی notices برای پیام خطا
                $notices = wc_get_notices( 'error' );
                $error_message = ! empty( $notices ) ? wp_strip_all_tags( $notices[0]['notice'] ) : __( 'Could not add product to cart.', 'woocommerce' );
                
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    error_log( 'DACP Add to Cart - Failed: cart_item_key=' . var_export( $cart_item_key, true ) );
                    error_log( 'DACP Add to Cart - Error notices: ' . print_r( $notices, true ) );
                }
                
                wc_clear_notices();
                
                // اگر خطا داشت، redirect به صفحه محصول
                wp_send_json( array(
                    'error'       => true,
                    'product_url' => apply_filters( 'woocommerce_cart_redirect_after_error', get_permalink( $product_id ), $product_id ),
                    'message'     => $error_message,
                ) );
            }
        } catch ( Exception $e ) {
            // Catch کردن exception ها از WooCommerce
            $error_message = $e->getMessage();
            
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( 'DACP Add to Cart - Exception: ' . $e->getMessage() );
                error_log( 'DACP Add to Cart - Exception trace: ' . $e->getTraceAsString() );
            }
            
            if ( empty( $error_message ) ) {
                $error_message = __( 'Could not add product to cart.', 'woocommerce' );
            }
            
            wp_send_json( array(
                'error'       => true,
                'product_url' => apply_filters( 'woocommerce_cart_redirect_after_error', get_permalink( $product_id ), $product_id ),
                'message'     => $error_message,
            ) );
        }
        
    } catch ( Exception $e ) {
        // Catch کردن exception ها از WooCommerce (برای خطاهای خارج از try-catch داخلی)
        $error_message = $e->getMessage();
        if ( empty( $error_message ) ) {
            $error_message = __( 'An error occurred while adding the product to cart.', 'woocommerce' );
        }
        
        wp_send_json( array(
            'error'       => true,
            'product_url' => isset( $product_id ) ? apply_filters( 'woocommerce_cart_redirect_after_error', get_permalink( $product_id ), $product_id ) : '',
            'message'     => $error_message,
        ) );
    }
}

// Fragment handler برای به‌روزرسانی خودکار سبد خرید
function dacp_cart_fragments($fragments) {

    /* ========= Badge تعداد سبد بالای آیکن ========= */
    ob_start();

    $count      = WC()->cart->get_cart_contents_count();
    $cart_items = WC()->cart->get_cart();
    ?>
    <span class="cart-counter-badge flex items-center justify-center bg-heading text-white absolute -top-2.5 xl:-top-3 rounded-full ltr:-right-2.5 ltr:xl:-right-3 rtl:-left-2.5 rtl:xl:-left-3 font-bold">
        <?php echo $count; ?>
    </span>
    <?php
    $fragments['.cart-counter-badge'] = ob_get_clean();


    /* ========= محتوای سبد خرید (لیست محصولات) ========= */
    ob_start();
    ?>
    <div class="w-full px-5 md:px-7 dima-cart-items">
        <?php
        if ( $count > 0 ) {
            foreach ( $cart_items as $cart_item_key => $cart_item ) {

                $_product   = apply_filters( 'woocommerce_cart_item_product', $cart_item['data'], $cart_item, $cart_item_key );
                $product_id = apply_filters( 'woocommerce_cart_item_product_id', $cart_item['product_id'], $cart_item, $cart_item_key );

                if ( $_product && $_product->exists() && $cart_item['quantity'] > 0 ) {

                    $product_permalink = apply_filters(
                        'woocommerce_cart_item_permalink',
                        $_product->is_visible() ? $_product->get_permalink( $cart_item ) : '',
                        $cart_item,
                        $cart_item_key
                    );
                    $thumbnail        = apply_filters( 'woocommerce_cart_item_thumbnail', $_product->get_image(), $cart_item, $cart_item_key );
                    $product_name     = apply_filters( 'woocommerce_cart_item_name', $_product->get_name(), $cart_item, $cart_item_key );
                    $product_price    = apply_filters( 'woocommerce_cart_item_price', WC()->cart->get_product_price( $_product ), $cart_item, $cart_item_key );
                    $product_subtotal = apply_filters( 'woocommerce_cart_item_subtotal', WC()->cart->get_product_subtotal( $_product, $cart_item['quantity'] ), $cart_item, $cart_item_key );
                    
                    // دریافت variation attributes برای نمایش
                    $variation_data = '';
                    if ( ! empty( $cart_item['variation'] ) && is_array( $cart_item['variation'] ) ) {
                        $variation_data = wc_get_formatted_variation( $cart_item['variation'], true );
                    }
                    ?>
                    <div class="group w-full h-auto flex justify-start items-center bg-white py-4 md:py-7 border-b border-gray-100 relative last:border-b-0 dima-cart-item"
                         data-cart-key="<?php echo esc_attr( $cart_item_key ); ?>">

                        <div class="relative flex flex-shrink-0 w-24 h-24 overflow-hidden bg-gray-200 rounded-md cursor-pointer md:w-28 md:h-28 ltr:mr-4 rtl:ml-4">
                            <?php if ( ! $product_permalink ) : ?>
                                <?php echo $thumbnail; ?>
                            <?php else : ?>
                                <a href="<?php echo esc_url( $product_permalink ); ?>">
                                    <?php echo $thumbnail; ?>
                                </a>
                            <?php endif; ?>

                            <div class="dima-remove-item absolute top-0 flex items-center justify-center w-full h-full transition duration-200 ease-in-out bg-black ltr:left-0 rtl:right-0 bg-opacity-30 md:bg-opacity-0 md:group-hover:bg-opacity-30"
                                 role="button"
                                 data-cart-key="<?php echo esc_attr( $cart_item_key ); ?>">
                                <svg stroke="currentColor" fill="currentColor" stroke-width="0" viewBox="0 0 512 512"
                                     class="relative text-2xl text-white transition duration-300 ease-in-out transform md:scale-0 md:opacity-0 md:group-hover:scale-100 md:group-hover:opacity-100"
                                     height="1em" width="1em" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M256 48C141.1 48 48 141.1 48 256s93.1 208 208 208 208-93.1 208-208S370.9 48 256 48zm52.7 283.3L256 278.6l-52.7 52.7c-6.2 6.2-16.4 6.2-22.6 0-3.1-3.1-4.7-7.2-4.7-11.3 0-4.1 1.6-8.2 4.7-11.3l52.7-52.7-52.7-52.7c-3.1-3.1-4.7-7.2-4.7-11.3 0-4.1 1.6-8.2 4.7-11.3 6.2-6.2 16.4-6.2 22.6 0l52.7 52.7 52.7-52.7c6.2-6.2 16.4-6.2 22.6 0 6.2 6.2 6.2 16.4 0 22.6L278.6 256l52.7 52.7c6.2 6.2 6.2 16.4 0 22.6-6.2 6.3-16.4 6.3-22.6 0z"></path>
                                </svg>
                            </div>
                        </div>

                        <div class="flex flex-col w-full overflow-hidden">
                            <?php if ( ! $product_permalink ) : ?>
                                <span class="truncate text-sm text-heading mb-1.5 -mt-1">
                                    <?php echo wp_kses_post( $product_name ); ?>
                                </span>
                            <?php else : ?>
                                <a class="truncate text-sm text-heading mb-1.5 -mt-1"
                                   href="<?php echo esc_url( $product_permalink ); ?>">
                                    <?php echo wp_kses_post( $product_name ); ?>
                                </a>
                            <?php endif; ?>

                            <span class="text-sm text-gray-400 mb-2.5">
                                قیمت واحد : &nbsp; <?php echo $product_price; ?>
                            </span>
                            <?php if ( $variation_data ) : ?>
                                <div class="text-xs text-gray-500 mb-2">
                                    <?php echo wp_kses_post( $variation_data ); ?>
                                </div>
                            <?php endif; ?>

                            <div class="flex items-end justify-between">
                                <div class="group flex items-center justify-between rounded-md overflow-hidden flex-shrink-0 h-8 md:h-9 shadow-navigation bg-heading">
                                    <button class="dima-decrease-qty flex items-center justify-center flex-shrink-0 h-full transition ease-in-out duration-300 focus:outline-none w-8 md:w-9 text-white bg-heading hover:bg-gray-600 focus:outline-none"
                                            data-cart-key="<?php echo esc_attr( $cart_item_key ); ?>">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="10px" height="2px" viewBox="0 0 12 1.5">
                                            <rect data-name="Rectangle 970" width="10px" height="2px" fill="currentColor"></rect>
                                        </svg>
                                    </button>
                                    <span class="dima-item-qty font-semibold flex items-center justify-center h-full transition-colors duration-250 ease-in-out cursor-default flex-shrink-0 text-sm text-white w-8 md:w-10">
                                        <?php echo esc_html( $cart_item['quantity'] ); ?>
                                    </span>
                                    <button class="dima-increase-qty flex items-center justify-center h-full flex-shrink-0 transition ease-in-out duration-300 focus:outline-none w-8 md:w-9 text-white bg-heading hover:bg-gray-600 focus:outline-none"
                                            data-cart-key="<?php echo esc_attr( $cart_item_key ); ?>">
                                        <svg data-name="plus (2)" xmlns="http://www.w3.org/2000/svg" width="10px" height="10px" viewBox="0 0 12 12">
                                            <g data-name="Group 5367">
                                                <path data-name="Path 17138" d="M6.749,5.251V0h-1.5V5.251H0v1.5H5.251V12h1.5V6.749H12v-1.5Z" fill="currentColor"></path>
                                            </g>
                                        </svg>
                                    </button>
                                </div>

                                <!-- جمع هر ردیف -->
                                <span class="dima-item-subtotal text-sm font-semibold leading-5 md:text-base text-heading">
                                    <?php echo $product_subtotal; ?>
                                </span>
                            </div>
                        </div>
                    </div>
                    <?php
                }
            }
        }
        ?>
    </div>
    <?php
    $fragments['.dima-cart-items'] = ob_get_clean();


    /* ========= جمع کل سبد (پایین دکمه) ========= */
    $fragments['.dima-cart-total'] = '<span class="dima-cart-total">' . WC()->cart->get_cart_total() . '</span>';

    return $fragments;
}

// AJAX Handler: Remove item from cart
function dacp_remove_cart_item() {
    // بررسی nonce
    if ( isset($_POST['nonce']) && ! wp_verify_nonce($_POST['nonce'], 'dacp_nonce') ) {
        wp_send_json_error(['message' => 'Invalid nonce']);
        return;
    }
    
    if ( ! isset($_POST['cart_key']) ) {
        wp_send_json_error(['message' => 'Cart key missing']);
        return;
    }
    
    $cart_key = sanitize_text_field($_POST['cart_key']);
    
    // بررسی اینکه cart key در سبد خرید وجود دارد
    $cart_contents = WC()->cart->get_cart();
    
    if ( ! isset($cart_contents[$cart_key]) ) {
        // اگر محصول وجود ندارد، فقط fragments را برگردان
        WC()->cart->calculate_totals();
        WC_AJAX::get_refreshed_fragments();
        wp_die();
    }
    
    // حذف محصول
    $removed = WC()->cart->remove_cart_item($cart_key);
    
    // محاسبه مجدد سبد خرید
    WC()->cart->calculate_totals();
    
    // بازگرداندن fragments
    WC_AJAX::get_refreshed_fragments();
    
    wp_die();
}

// AJAX Handler: Update item quantity (با چک موجودی و فروش تکی)
function dacp_update_cart_qty() {
    // بررسی nonce
    if ( isset($_POST['nonce']) && ! wp_verify_nonce($_POST['nonce'], 'dacp_nonce') ) {
        wp_send_json_error(['message' => 'Invalid nonce']);
        return;
    }

    if ( ! isset($_POST['cart_key']) || ! isset($_POST['quantity']) ) {
        wp_send_json_error(['message' => 'Missing parameters']);
        return;
    }

    $cart_key = sanitize_text_field( $_POST['cart_key'] );
    $quantity = intval( $_POST['quantity'] );

    // جلوگیری از مقدار منفی
    if ( $quantity < 0 ) {
        $quantity = 0;
    }

    $cart          = WC()->cart;
    $cart_contents = $cart->get_cart();

    if ( ! isset( $cart_contents[ $cart_key ] ) ) {
        WC()->cart->calculate_totals();
        WC_AJAX::get_refreshed_fragments();
        wp_die();
    }

    $cart_item = $cart_contents[ $cart_key ];
    /** @var WC_Product $_product */
    $_product = $cart_item['data'];

    // 🔹 اگر محصول فروش تکی است (Sold Individually)
    if ( $_product->is_sold_individually() ) {
        if ( $quantity > 1 ) {
            $quantity = 1;
        }
    }

    // 🔹 بررسی موجودی انبار
    if ( $quantity > 0 && $_product->managing_stock() && ! $_product->backorders_allowed() ) {
        $stock_qty = $_product->get_stock_quantity();

        // اگر موجودی صفر بود → حذف کن
        if ( $stock_qty <= 0 ) {
            $cart->remove_cart_item( $cart_key );
            $cart->calculate_totals();
            WC_AJAX::get_refreshed_fragments();
            wp_die();
        }

        // اگر بیشتر از موجودی درخواست شد → محدود کن
        if ( $quantity > $stock_qty ) {
            $quantity = $stock_qty;
        }
    }

    // 🔹 اگر بعد از همه چک‌ها تعداد شد صفر → حذف کن
    if ( $quantity <= 0 ) {
        $cart->remove_cart_item( $cart_key );
    } else {
        $cart->set_quantity( $cart_key, $quantity, true );
    }

    // 🔹 به‌روزرسانی و ارسال فرگمنت‌ها
    $cart->calculate_totals();
    WC_AJAX::get_refreshed_fragments();

    wp_die();
}

// Render button
function dacp_render_button() {
    if ( ! WC()->cart ) return '';
    $count = WC()->cart->get_cart_contents_count();
    ob_start();
    ?>
    <?php
    return ob_get_clean();
}

// Render drawer
function dacp_render_drawer() {
    if ( ! WC()->cart ) return '';
    ob_start();
    ?>
    <?php
    return ob_get_clean();
}

// Widget class
class Dima_Ajax_Cart_Pro_Widget extends WP_Widget {
    public function __construct() {
        parent::__construct('dima_ajax_cart_pro','Dima Ajax Cart Pro',[ 'description'=>'Tailwind-friendly Woo mini cart toggle' ]);
    }
    
    public function widget( $args, $instance ) {
    $count = WC()->cart->get_cart_contents_count();
    $cart_items = WC()->cart->get_cart();
    $cart_total = WC()->cart->get_cart_total();
    $checkout_url = wc_get_checkout_url();
    $cart_url = wc_get_cart_url();
?>
<button class="dacp-btn relative flex items-center justify-center flex-shrink-0 h-auto transform focus:outline-none" aria-label="cart-button">
	<svg xmlns="http://www.w3.org/2000/svg" width="18px" height="18px" viewBox="0 0 20 20" class="md:w-4 xl:w-5 md:h-4 xl:h-5">
		<path d="M5,4H19a1,1,0,0,1,1,1V19a1,1,0,0,1-1,1H5a1,1,0,0,1-1-1V5A1,1,0,0,1,5,4ZM2,5A3,3,0,0,1,5,2H19a3,3,0,0,1,3,3V19a3,3,0,0,1-3,3H5a3,3,0,0,1-3-3Zm10,7C9.239,12,7,9.314,7,6H9c0,2.566,1.669,4,3,4s3-1.434,3-4h2C17,9.314,14.761,12,12,12Z" transform="translate(-2 -2)" fill="currentColor" fill-rule="evenodd"></path>
	</svg>
	<span class="cart-counter-badge flex items-center justify-center bg-heading text-white absolute -top-2.5 xl:-top-3 rounded-full ltr:-right-2.5 ltr:xl:-right-3 rtl:-left-2.5 rtl:xl:-left-3 font-bold"><?php echo $count; ?></span>
</button>
<div id="dacp-overlay" class="dacp-overlay"></div>
<div id="dacp-drawer" class="rc-drawer rc-drawer-left cart-drawer" tabindex="-1">
	<div tabindex="0" aria-hidden="true" data-sentinel="start"></div>
	<div class="rc-drawer-content-wrapper rc-drawer-content-wrapper-hidden">
		<div class="rc-drawer-content" aria-modal="true" role="dialog">
			<div class="flex flex-col justify-between w-full h-screen">
				<div class="w-full flex justify-between items-center relative ltr:pl-5 ltr:md:pl-7 rtl:pr-5 rtl:md:pr-7 py-0.5 border-b border-gray-100">
					<h2 class="m-0 text-xl font-bold md:text-2xl text-heading">سبد خرید</h2>
					<button class="dacp-close flex items-center justify-center px-4 py-6 text-2xl text-gray-500 transition-opacity md:px-6 lg:py-8 focus:outline-none hover:opacity-60" aria-label="close" data-cart-close>
						<svg stroke="currentColor" fill="currentColor" stroke-width="0" viewBox="0 0 512 512" class="text-black mt-1 md:mt-0.5" height="1em" width="1em" xmlns="http://www.w3.org/2000/svg">
							<path d="M289.94 256l95-95A24 24 0 00351 127l-95 95-95-95a24 24 0 00-34 34l95 95-95 95a24 24 0 1034 34l95-95 95 95a24 24 0 0034-34z"></path>
						</svg>
					</button>
				</div>
				<?php if($count):?>
                                <div class="flex-grow min-h-0 w-full overflow-y-auto cart-scrollbar">
                                    <div class="w-full px-5 md:px-7 dima-cart-items">
                                        <?php
                                        foreach ( $cart_items as $cart_item_key => $cart_item ) {
                                            $_product = apply_filters( 'woocommerce_cart_item_product', $cart_item['data'], $cart_item, $cart_item_key );
                                            $product_id = apply_filters( 'woocommerce_cart_item_product_id', $cart_item['product_id'], $cart_item, $cart_item_key );
                                            
                                            if ( $_product && $_product->exists() && $cart_item['quantity'] > 0 ) {
                                                $product_permalink = apply_filters( 'woocommerce_cart_item_permalink', $_product->is_visible() ? $_product->get_permalink( $cart_item ) : '', $cart_item, $cart_item_key );
                                                $thumbnail = apply_filters( 'woocommerce_cart_item_thumbnail', $_product->get_image(), $cart_item, $cart_item_key );
                                                $product_name = apply_filters( 'woocommerce_cart_item_name', $_product->get_name(), $cart_item, $cart_item_key );
                                                $product_price = apply_filters( 'woocommerce_cart_item_price', WC()->cart->get_product_price( $_product ), $cart_item, $cart_item_key );
                                                $product_subtotal = apply_filters( 'woocommerce_cart_item_subtotal', WC()->cart->get_product_subtotal( $_product, $cart_item['quantity'] ), $cart_item, $cart_item_key );
                                                
                                                // دریافت variation attributes برای نمایش
                                                $variation_data = '';
                                                if ( ! empty( $cart_item['variation'] ) && is_array( $cart_item['variation'] ) ) {
                                                    $variation_data = wc_get_formatted_variation( $cart_item['variation'], true );
                                                }
                                                ?>
                                                <div class="group w-full h-auto flex justify-start items-center bg-white py-4 md:py-7 border-b border-gray-100 relative last:border-b-0 dima-cart-item" data-cart-key="<?php echo esc_attr( $cart_item_key ); ?>">
                                                    <div class="relative flex flex-shrink-0 w-24 h-24 overflow-hidden bg-gray-200 rounded-md cursor-pointer md:w-28 md:h-28 ltr:mr-4 rtl:ml-4">
                                                        <?php if ( ! $product_permalink ) : ?>
                                                            <?php echo $thumbnail; ?>
                                                        <?php else : ?>
                                                            <a href="<?php echo esc_url( $product_permalink ); ?>"><?php echo $thumbnail; ?></a>
                                                        <?php endif; ?>
                                                        <div class="dima-remove-item absolute top-0 flex items-center justify-center w-full h-full transition duration-200 ease-in-out bg-black ltr:left-0 rtl:right-0 bg-opacity-30 md:bg-opacity-0 md:group-hover:bg-opacity-30" role="button" data-cart-key="<?php echo esc_attr( $cart_item_key ); ?>">
                                                            <svg stroke="currentColor" fill="currentColor" stroke-width="0" viewBox="0 0 512 512" class="relative text-2xl text-white transition duration-300 ease-in-out transform md:scale-0 md:opacity-0 md:group-hover:scale-100 md:group-hover:opacity-100" height="1em" width="1em" xmlns="http://www.w3.org/2000/svg">
                                                                <path d="M256 48C141.1 48 48 141.1 48 256s93.1 208 208 208 208-93.1 208-208S370.9 48 256 48zm52.7 283.3L256 278.6l-52.7 52.7c-6.2 6.2-16.4 6.2-22.6 0-3.1-3.1-4.7-7.2-4.7-11.3 0-4.1 1.6-8.2 4.7-11.3l52.7-52.7-52.7-52.7c-3.1-3.1-4.7-7.2-4.7-11.3 0-4.1 1.6-8.2 4.7-11.3 6.2-6.2 16.4-6.2 22.6 0l52.7 52.7 52.7-52.7c6.2-6.2 16.4-6.2 22.6 0 6.2 6.2 6.2 16.4 0 22.6L278.6 256l52.7 52.7c6.2 6.2 6.2 16.4 0 22.6-6.2 6.3-16.4 6.3-22.6 0z"></path>
                                                            </svg>
                                                        </div>
                                                    </div>
                                                    <div class="flex flex-col w-full overflow-hidden">
                                                        <?php if ( ! $product_permalink ) : ?>
                                                            <span class="truncate text-sm text-heading mb-1.5 -mt-1"><?php echo wp_kses_post( $product_name ); ?></span>
                                                        <?php else : ?>
                                                            <a class="truncate text-sm text-heading mb-1.5 -mt-1" href="<?php echo esc_url( $product_permalink ); ?>"><?php echo wp_kses_post( $product_name ); ?></a>
                                                        <?php endif; ?>
                                                        <span class="text-sm text-gray-400 mb-2.5">قیمت واحد : &nbsp; <?php echo $product_price; ?></span>
                                                        <?php if ( $variation_data ) : ?>
                                                            <div class="text-xs text-gray-500 mb-2">
                                                                <?php echo wp_kses_post( $variation_data ); ?>
                                                            </div>
                                                        <?php endif; ?>
                                                        <div class="flex items-end justify-between">
                                                            <div class="group flex items-center justify-between rounded-md overflow-hidden flex-shrink-0 h-8 md:h-9 shadow-navigation bg-heading">
                                                                <button class="dima-decrease-qty flex items-center justify-center flex-shrink-0 h-full transition ease-in-out duration-300 focus:outline-none w-8 md:w-9 text-white bg-heading hover:bg-gray-600 focus:outline-none" data-cart-key="<?php echo esc_attr( $cart_item_key ); ?>">
                                                                    <svg xmlns="http://www.w3.org/2000/svg" width="10px" height="2px" viewBox="0 0 12 1.5">
                                                                        <rect data-name="Rectangle 970" width="10px" height="2px" fill="currentColor"></rect>
                                                                    </svg>
                                                                </button>
                                                                <span class="dima-item-qty font-semibold flex items-center justify-center h-full transition-colors duration-250 ease-in-out cursor-default flex-shrink-0 text-sm text-white w-8 md:w-10"><?php echo esc_html( $cart_item['quantity'] ); ?></span>
                                                                <button class="dima-increase-qty flex items-center justify-center h-full flex-shrink-0 transition ease-in-out duration-300 focus:outline-none w-8 md:w-9 text-white bg-heading hover:bg-gray-600 focus:outline-none" data-cart-key="<?php echo esc_attr( $cart_item_key ); ?>">
                                                                    <svg data-name="plus (2)" xmlns="http://www.w3.org/2000/svg" width="10px" height="10px" viewBox="0 0 12 12">
                                                                        <g data-name="Group 5367">
                                                                            <path data-name="Path 17138" d="M6.749,5.251V0h-1.5V5.251H0v1.5H5.251V12h1.5V6.749H12v-1.5Z" fill="currentColor"></path>
                                                                        </g>
                                                                    </svg>
                                                                </button>
                                                            </div>
                                                            <span class="dima-item-subtotal text-sm font-semibold leading-5 md:text-base text-heading"><?php echo $product_subtotal; ?></span>
                                                        </div>
                                                    </div>
                                                </div>
                                                <?php
                                            }
                                        }
                                        ?>
                                    </div>
                                </div>
                                
                                <!-- Footer with checkout button -->
                                <div class="flex flex-col px-5 pt-2 pb-5 md:px-7 md:pb-7">
                                    <a class="w-full px-5 py-3 md:py-4 flex items-center justify-center rounded-md text-sm sm:text-base text-white focus:outline-none transition duration-300 bg-heading hover:bg-gray-600 dima-checkout-btn" href="<?php echo esc_url( $checkout_url ); ?>">
                                        <span class="w-full ltr:pr-5 rtl:pl-5 -mt-0.5 py-0.5">تسویه حساب </span>
                                        <span class="rtl:mr-auto ltr:ml-auto flex-shrink-0 -mt-0.5 py-0.5 flex">
                                            <span class="ltr:border-l rtl:border-r border-white ltr:pr-5 rtl:pl-5 py-0.5"></span>
                                            <span class="dima-cart-total"><?php echo $cart_total; ?></span>
                                        </span>
                                    </a>
                                </div>
				<?php else:?>
				<div class="flex flex-col items-center justify-center px-5 pt-8 pb-5 md:px-7">
                     <div class="flex flex-col items-center justify-center px-5 pt-8 pb-5 md:px-7" style="opacity: 1; transform: none; transform-origin: 50% 50% 0px;">
								<svg width="180" height="309.722" viewBox="0 0 180 309.722">
									<g id="no_cart_in_bag" data-name="no cart in bag" transform="translate(-988 -7673)">
										<g id="no_cart" data-name="no cart" transform="translate(988 7673)">
											<g id="Group_5970" data-name="Group 5970" transform="translate(0 0)">
												<g id="Group_5967" data-name="Group 5967" transform="translate(17.408 92.119)">
													<path id="Path_17743" data-name="Path 17743" d="M405.506,794.581l63.621-36.762L418.2,724.274Z" transform="translate(-323.428 -576.978)" fill="#ba8064"></path>
													<path id="Path_17744" data-name="Path 17744" d="M135.711,140.727l32.918-.12,1.287-20.225,79.451,45.843-34.42,1.084v19.165Z" transform="translate(-118.988 -119.373)" fill="#dba480"></path>
													<path id="Path_17745" data-name="Path 17745" d="M314.4,206.341,272,124.761l-2.279,22.008,1.4,59.572Z" transform="translate(-220.537 -122.691)" fill="#460100" opacity="0.1"></path>
													<path id="Path_17746" data-name="Path 17746" d="M141.237,253.056l-10.26-47.388,34.59-.126v37.243Z" transform="translate(-115.402 -183.904)" fill="#995b47"></path>
													<path id="Path_17747" data-name="Path 17747" d="M511.029,445.295l-49.136-22.179L460.8,385.489l1.089-72.515,35.954-1.133Z" transform="translate(-365.33 -264.454)" fill="#a96e56"></path>
													<path id="Path_17748" data-name="Path 17748" d="M148.755,398.756l9.58-70.307,4.9-79.149L81.161,201.911,66.677,351.368Z" transform="translate(-66.677 -181.153)" fill="#dba480"></path>
													<path id="Path_17749" data-name="Path 17749" d="M349.459,429.379c-.415,1.942-2.182,2.6-3.948,1.46a5.753,5.753,0,0,1-2.446-5.572c.414-1.942,2.182-2.6,3.948-1.46A5.753,5.753,0,0,1,349.459,429.379Z" transform="translate(-276.046 -348.874)" fill="#67251d"></path>
													<path id="Path_17750" data-name="Path 17750" d="M209.819,348.753c-.415,1.942-2.182,2.6-3.948,1.46a5.753,5.753,0,0,1-2.446-5.572c.415-1.942,2.182-2.6,3.948-1.46A5.753,5.753,0,0,1,209.819,348.753Z" transform="translate(-170.233 -287.779)" fill="#67251d"></path>
													<g id="Group_5965" data-name="Group 5965" transform="translate(31.503 60.166)" opacity="0.2">
														<path id="Path_17751" data-name="Path 17751" d="M219.35,441.507a16.861,16.861,0,0,1-8.439-2.272A28.35,28.35,0,0,1,196.858,412l4.383-45.226a2.414,2.414,0,0,1,4.806.467l-4.383,45.226a23.483,23.483,0,0,0,11.608,22.554,12.055,12.055,0,0,0,18.081-9.247l3.819-39.41a2.414,2.414,0,0,1,4.806.467l-3.819,39.41a16.912,16.912,0,0,1-16.809,15.266Z" transform="translate(-196.727 -364.591)" fill="#460100"></path>
													</g>
													<path id="Path_17752" data-name="Path 17752" d="M162.373,116.218,161.06,136.85l-34.59.126,82.078,47.388V164.738l35.954-1.132Zm44.968,47.351v18.7l-76.395-44.106,31.247-.113,1.261-19.819,76.774,44.3Z" transform="translate(-111.986 -116.218)" fill="#fcc89d"></path>
													<g id="Group_5966" data-name="Group 5966" transform="translate(29.24 57.45)">
														<path id="Path_17753" data-name="Path 17753" d="M210.007,430.3a16.864,16.864,0,0,1-8.438-2.271,28.35,28.35,0,0,1-14.054-27.235l4.383-45.226a2.414,2.414,0,0,1,4.806.467l-4.383,45.226a23.483,23.483,0,0,0,11.608,22.554,12.055,12.055,0,0,0,18.081-9.247l3.819-39.41a2.414,2.414,0,0,1,4.806.467l-3.819,39.41A16.912,16.912,0,0,1,210.007,430.3Z" transform="translate(-187.384 -353.38)" fill="#995b47"></path>
													</g>
													<path id="Path_17754" data-name="Path 17754" d="M405.506,546.991,419.99,488.05V397.534Z" transform="translate(-323.428 -329.388)" fill="#995b47"></path>
												</g>
												<g id="Group_5968" data-name="Group 5968" transform="translate(0 0)">
													<path id="Path_17755" data-name="Path 17755" d="M394.573,120.6c-.142-.5.244-.855.5-1.088a1.4,1.4,0,0,0,.271-.293,0,0,0,0,0,0,0,1.39,1.39,0,0,0-.384-.107c-.34-.065-.853-.162-1-.665s.244-.855.5-1.088a1.39,1.39,0,0,0,.271-.293,0,0,0,0,0,0,0,1.4,1.4,0,0,0-.384-.107c-.34-.064-.853-.162-1-.664s.244-.855.5-1.088l.009-.008a.9.9,0,0,0,.259-.482.391.391,0,0,1,.276-.292.41.41,0,0,1,.5.316,1.122,1.122,0,0,1-.51,1.046,1.4,1.4,0,0,0-.271.292,0,0,0,0,0,0,0,1.4,1.4,0,0,0,.384.107c.34.065.853.162,1,.665s-.244.855-.5,1.088a1.4,1.4,0,0,0-.271.293,0,0,0,0,0,0,0,1.391,1.391,0,0,0,.384.107c.34.065.853.162,1,.665s-.244.855-.5,1.088a1.257,1.257,0,0,0-.273.3,0,0,0,0,0,0,0,1.641,1.641,0,0,0,.387.1c.331.063.826.157.983.625a.416.416,0,0,1-.21.507.392.392,0,0,1-.456-.109.789.789,0,0,0-.464-.253h0C395.229,121.2,394.716,121.1,394.573,120.6Z" transform="translate(-349.075 -37.518)" fill="#212121" opacity="0.3"></path>
													<path id="Path_17765" data-name="Path 17765" d="M395.468,120.6c.142-.5-.244-.855-.5-1.088a1.4,1.4,0,0,1-.271-.293,0,0,0,0,1,0,0,1.39,1.39,0,0,1,.384-.107c.34-.065.853-.162,1-.665s-.244-.855-.5-1.088a1.389,1.389,0,0,1-.271-.293,0,0,0,0,1,0,0,1.4,1.4,0,0,1,.384-.107c.34-.064.853-.162,1-.664s-.244-.855-.5-1.088l-.009-.008a.9.9,0,0,1-.259-.482.391.391,0,0,0-.276-.292.41.41,0,0,0-.5.316,1.122,1.122,0,0,0,.51,1.046,1.4,1.4,0,0,1,.271.292,0,0,0,0,1,0,0,1.4,1.4,0,0,1-.384.107c-.34.065-.853.162-1,.664s.244.855.5,1.088a1.4,1.4,0,0,1,.271.293,0,0,0,0,1,0,0,1.39,1.39,0,0,1-.384.107c-.34.065-.853.162-1,.665s.244.855.5,1.088a1.257,1.257,0,0,1,.273.3,0,0,0,0,1,0,0,1.641,1.641,0,0,1-.387.1c-.331.063-.826.157-.983.625a.416.416,0,0,0,.21.507.392.392,0,0,0,.456-.109.789.789,0,0,1,.464-.253h0C394.812,121.2,395.326,121.1,395.468,120.6Z" transform="translate(-262.76 -23.736)" fill="#212121" opacity="0.3"></path>
													<path id="Path_17756" data-name="Path 17756" d="M375.447,179.277a2.539,2.539,0,1,1,3.346,1.3A2.542,2.542,0,0,1,375.447,179.277Zm3.737-1.643a1.543,1.543,0,1,0-.791,2.034A1.545,1.545,0,0,0,379.184,177.634Z" transform="translate(-375.232 -52.408)" fill="#212121" opacity="0.3"></path>
													<path id="Path_17764" data-name="Path 17764" d="M375.447,179.277a2.539,2.539,0,1,1,3.346,1.3A2.542,2.542,0,0,1,375.447,179.277Zm3.737-1.643a1.543,1.543,0,1,0-.791,2.034A1.545,1.545,0,0,0,379.184,177.634Z" transform="translate(-333.888 -175.716)" fill="#212121" opacity="0.3"></path>
													<path id="Path_17757" data-name="Path 17757" d="M350.086,264.8a1.852,1.852,0,0,1-2.682-2.547l-.868-.823a3.047,3.047,0,0,0,4.417,4.194Z" transform="translate(-253.642 -206.302)" fill="#212121" opacity="0.3"></path>
													<path id="Path_17766" data-name="Path 17766" d="M346.628,264.8a1.852,1.852,0,0,0,2.682-2.547l.867-.823a3.047,3.047,0,0,1-4.417,4.194Z" transform="translate(-170.953 -110.557)" fill="#212121" opacity="0.15"></path>
													<path id="Union_11" data-name="Union 11" d="M2.059,6.97l.989-3.048L0,2.933.283,2.06l3.049.989L4.321,0,5.2.284l-.99,3.048,3.047.989L6.97,5.2l-3.048-.99-.99,3.049Z" transform="translate(6.528 48.598)" fill="#212121" stroke="rgba(0,0,0,0)" stroke-miterlimit="10" stroke-width="1" opacity="0.3"></path>
													<path id="Union_13" data-name="Union 13" d="M2.059,6.97l.989-3.048L0,2.933.283,2.06l3.049.989L4.321,0,5.2.284l-.99,3.048,3.047.989L6.97,5.2l-3.048-.99-.99,3.049Z" transform="translate(94.294 121.132)" fill="#212121" stroke="rgba(0,0,0,0)" stroke-miterlimit="10" stroke-width="1" opacity="0.3"></path>
													<path id="Union_12" data-name="Union 12" d="M1.235,4.182l.593-1.829L0,1.759l.17-.524L2,1.829,2.592,0l.525.17L2.523,2l1.828.594-.17.523L2.353,2.523,1.759,4.352Z" transform="translate(107.351 6.528)" fill="#212121" stroke="rgba(0,0,0,0)" stroke-miterlimit="10" stroke-width="1" opacity="0.3"></path>
												</g>
											</g>
										</g>
									</g>
								</svg>
								<h3 class="pt-8 text-lg font-bold text-heading">سبد خرید خالی است</h3>
							</div>				   
				</div>
				<div class="flex flex-col px-5 pt-2 pb-5 md:px-7 md:pb-7">
               <a class="w-full px-5 py-3 md:py-4 flex items-center justify-center rounded-md text-sm sm:text-base text-white focus:outline-none transition duration-300  cursor-not-allowed bg-gray-400 hover:bg-gray-400" href="<?php echo esc_url( $cart_url ); ?>">
                  <span class="w-full ltr:pr-5 rtl:pl-5 -mt-0.5 py-0.5">جمع کل</span>
                  <span class="rtl:mr-auto ltr:ml-auto flex-shrink-0 -mt-0.5 py-0.5 flex"><span class="ltr:border-l rtl:border-r border-white ltr:pr-5 rtl:pl-5 py-0.5"> </span>۰ ریال</span>
               </a>
				</div>
				<?php endif;?>
			</div>
		</div>
	</div>
	<div tabindex="0" aria-hidden="true" data-sentinel="end"></div>
</div>
<?php
    }
    
    public function form( $instance ) {
        $title = isset( $instance['title'] ) ? $instance['title'] : '';
        ?>
        <p>
            <label for="<?php echo esc_attr( $this->get_field_id('title') ); ?>">عنوان:</label>
            <input class="widefat" id="<?php echo esc_attr( $this->get_field_id('title') ); ?>" name="<?php echo esc_attr( $this->get_field_name('title') ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
        </p>
        <?php
    }
    
    public function update( $new_instance, $old_instance ) {
        $instance = [];
        $instance['title'] = isset($new_instance['title']) ? sanitize_text_field($new_instance['title']) : '';
        return $instance;
    }
}